<?php

namespace App\Http\Controllers;

use App\Services\BranchService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Requests\{ BranchRequest, UpdateBranchRequest };
use Inertia\Inertia;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\BranchesExport;
use Maatwebsite\Excel\Facades\Excel;

class BranchController extends Controller
{
    protected $branchService;

    public function __construct(BranchService $branchService)
    {
        $this->branchService = $branchService;
    }

    public function index()
    {
        return Inertia::render('Admin/Branches/Index');
    }

    /**
     * Api branches list and search branches.
     */
    public function list(Request $request)
    {
        if ($request->query('all')) {
            return response()->json($this->branchService->getAll()); // Todas las sucursales sin paginar
        }

        return $this->branchService->list($request);
    }

    public function create()
    {
        return Inertia::render('Admin/Branches/Form');
    }

    public function show($id): JsonResponse
    {
        $branch = $this->branchService->getById($id);
        return $branch ? response()->json($branch) : response()->json(['message' => 'Not found'], 404);
    }

    public function store(BranchRequest $request): JsonResponse
    {
        return response()->json(
            [
                'message' => 'Sucursal registrada correctamente.',
                'branch' => $this->branchService->create($request->validated())
            ], 201
        );
    }

    public function edit($id)
    {
        $branch = $this->branchService->getById($id);

        return Inertia::render('Admin/Branches/Form', [
            'branch' => $branch
        ]);
    }

    public function update(UpdateBranchRequest $request, $id): JsonResponse
    {
        $branch = $this->branchService->update($id, $request->validated());

        if ($branch) {
            return response()->json(
                [
                    'message' => 'Sucursal actualizada correctamente.',
                    'branch' => $branch
                ], 201
            );
        }

        return response()->json(['message' => 'Not found'], 404);
    }

    public function destroy($id): JsonResponse
    {
        return $this->branchService->delete($id)
            ? response()->json(['message' => 'Sucursal eliminada correctamente.'])
            : response()->json(['message' => 'Not found'], 404);
    }

    public function exportPDF(Request $request)
    {
        $branches = $this->branchService->list($request);
        $pdf = Pdf::loadView('pdf.branches', ['branches' => $branches]);
        return $pdf->stream('branches.pdf');
    }

    public function exportExcel(Request $request)
    {
        $branches = $this->branchService->list($request);
        return Excel::download(new BranchesExport($branches), 'reporte-sucursales.xlsx');
    }
}
