<?php

namespace App\Http\Controllers;

use App\Services\BrandService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Requests\{ BrandRequest, UpdateBrandRequest };
use Inertia\Inertia;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\BrandsExport;
use Maatwebsite\Excel\Facades\Excel;

class BrandController extends Controller
{
    protected $brandService;

    public function __construct(BrandService $brandService)
    {
        $this->brandService = $brandService;
    }

    public function index()
    {
        return Inertia::render('Admin/Brands/Index');
    }

    /**
     * Api brands list and search brands.
     */
    public function list(Request $request)
    {
        if ($request->query('all')) {
            return response()->json($this->brandService->getAll()); // todas las marcas
        }

        return $this->brandService->list($request);
    }

    public function create()
    {
        return Inertia::render('Admin/Brands/Form');
    }

    public function show($id): JsonResponse
    {
        $brand = $this->brandService->getById($id);
        return $brand 
            ? response()->json($brand) 
            : response()->json(['message' => 'Not found'], 404);
    }

    public function store(BrandRequest $request): JsonResponse
    {
        // aquí se guarda con el usuario logueado
        $data = $request->validated();
        $data['user_id'] = auth()->id();

        return response()->json(
            [
                'message' => 'Marca registrada correctamente.',
                'brand' => $this->brandService->create($data)
            ], 201
        );
    }

    public function edit($id)
    {
        $brand = $this->brandService->getById($id);

        return Inertia::render('Admin/Brands/Form', [
            'brand' => $brand
        ]);
    }

    public function update(UpdateBrandRequest $request, $id): JsonResponse
    {
        $brand = $this->brandService->update($id, $request->validated());

        if ($brand) {
            return response()->json(
                [
                    'message' => 'Marca actualizada correctamente.',
                    'brand' => $brand
                ], 201
            );
        }
        
        return response()->json(['message' => 'Not found'], 404);
    }

    public function destroy($id): JsonResponse
    {
        return $this->brandService->delete($id)
            ? response()->json(['message' => 'Marca eliminada correctamente.'])
            : response()->json(['message' => 'Not found'], 404);
    }

    public function exportPDF(Request $request)
    {
        $brands = $this->brandService->list($request);
        $pdf = Pdf::loadView('pdf.brands', ['brands' => $brands]);
        return $pdf->stream('brands.pdf');
    }

    public function exportExcel(Request $request)
    {
        $brands = $this->brandService->list($request);
        return Excel::download(new BrandsExport($brands), 'reporte-marcas.xlsx');
    }
}
