<?php

namespace App\Http\Controllers;

use App\Services\{ CashRegisterService, SaleService };
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Requests\{ OpenCashRegisterRequest, UpdateCashRegisterRequest };
use Inertia\Inertia;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\CashRegistersExport;
use Maatwebsite\Excel\Facades\Excel;

use App\Helpers\UserHelper;

class CashRegisterController extends Controller
{
    protected $cashRegisterService;
    protected $saleService;

    public function __construct(CashRegisterService $cashRegisterService, SaleService $saleService)
    {
        $this->cashRegisterService = $cashRegisterService;
        $this->saleService = $saleService;
    }

    public function index()
    {
        return Inertia::render('Admin/CashRegisters/Index');
    }

    /**
     * Api cash register list and search.
     */
    public function list(Request $request)
    {
        if ($request->query('all')) {
            return response()->json($this->cashRegisterService->getAll()); 
        }

        return $this->cashRegisterService->list($request);
    }

    public function create()
    {
        return Inertia::render('Admin/CashRegisters/Form');
    }

    public function show($id): JsonResponse
    {
        $cashRegister = $this->cashRegisterService->getById($id);

        if (!$cashRegister) {
            return response()->json(['message' => 'Caja no encontrada.'], 404);
        }

        return response()->json($cashRegister);
    }

    public function store(OpenCashRegisterRequest $request): JsonResponse
    {
        $branchId = UserHelper::branchId($request->user_id);

        $request["branch_id"] = $branchId;

        $cashRegister = $this->cashRegisterService->open($request);

        return response()->json(
            [
                'message' => 'Caja registrada correctamente.',
                'cash_register' => $cashRegister
            ],
            201
        );
    }

    public function close($id, $userId): JsonResponse
    {
        $cashRegister = $this->cashRegisterService->getById($id);

        if ($cashRegister->status == "closed") {
            return response()->json(['message' => 'La caja ya está cerrada.'], 400);
        }

        $totalAmountClose = $this->saleService->calculateTotalSale($id);
        $cashRegister = $this->cashRegisterService->close($id, $userId, $totalAmountClose);

        return response()->json(
            [
                'message' => 'Caja cerrada correctamente.',
                'cash_register' => $cashRegister
            ],
            200
        );
    }

    public function exportPDF(Request $request)
    {
        $cashRegisters = $this->cashRegisterService->getAll();
        $pdf = Pdf::loadView('pdf.cash_registers', ['cashRegisters' => $cashRegisters]);
        return $pdf->stream('cash-registers.pdf');
    }

    public function exportExcel(Request $request)
    {
        $cashRegisters = $this->cashRegisterService->getAll();
        return Excel::download(new CashRegistersExport($cashRegisters), 'reporte-cajas.xlsx');
    }
}
