<?php

namespace App\Http\Controllers;

use App\Services\CategoryService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Requests\{ CategoryRequest, UpdateCategoryRequest };
use Inertia\Inertia;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\CategoriesExport;
use Maatwebsite\Excel\Facades\Excel;

class CategoryController extends Controller
{
    protected $categoryService;

    public function __construct(CategoryService $categoryService)
    {
        $this->categoryService = $categoryService;
    }

    public function index()
    {
        return Inertia::render('Admin/Categories/Index');
    }

    /**
     * Api categories list and search categories.
     *
     * @return \Illuminate\Http\Response
     */
    public function list(Request $request)
    {
        if ($request->query('all')) {
            return response()->json($this->categoryService->getAll()); // Retorna todas las categorías sin paginación
        }

        return $this->categoryService->list($request);
    }

    public function create()
    {
        return Inertia::render('Admin/Categories/Form');
    }

    public function show($id): JsonResponse
    {
        $category = $this->categoryService->getById($id);
        return $category ? response()->json($category) : response()->json(['message' => 'Not found'], 404);
    }

    public function store(CategoryRequest $request): JsonResponse
    {
        return response()->json(
            [
                'message' => 'Categoría registrada correctamente.',
                'category' => $this->categoryService->create($request->validated())
            ], 201
        );
    }

    public function edit($id)
    {
        $category = $this->categoryService->getById($id);

        return Inertia::render('Admin/Categories/Form', [
            'category' => $category
        ]);
    }

    public function update(UpdateCategoryRequest $request, $id): JsonResponse
    {
        $category = $this->categoryService->update($id, $request->validated());

        if ($category) {
            return response()->json(
                [
                    'message' => 'Categoría actualizada correctamente.',
                    'category' => $category
                ], 201
            );
        }
        
        return response()->json(['message' => 'Not found'], 404);
    }

    public function destroy($id): JsonResponse
    {
        return $this->categoryService->delete($id)
            ? response()->json(['message' => 'Categoría eliminada correctamente.'])
            : response()->json(['message' => 'Not found'], 404);
    }

    public function exportPDF(Request $request)
    {
        $categories = $this->categoryService->list($request);
        $pdf = Pdf::loadView('pdf.categories', ['categories' => $categories]);
        return $pdf->stream('categories.pdf');
    }

    public function exportExcel(Request $request)
    {
        $categories = $this->categoryService->list($request);
        return Excel::download(new CategoriesExport($categories), 'reporte-categorias.xlsx');
    }
}
