<?php

namespace App\Http\Controllers;

use App\Http\Requests\{ ClientRequest, UpdateClientRequest };
use App\Models\Client;
use App\Services\ClientService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\ClientsExport;
use Maatwebsite\Excel\Facades\Excel;

class ClientController extends Controller
{
    protected $clientService;

    public function __construct(ClientService $clientService)
    {
        $this->clientService = $clientService;
    }

    public function index()
    {
        return Inertia::render('Admin/Clients/Index');
    }

    /**
     * Api categories list and search clients.
     *
     * @return \Illuminate\Http\Response
     */
    public function list(Request $request)
    {
        if ($request->query('all')) {
            // Retorna todos los clientes sin paginación
            return response()->json($this->clientService->getAll()); 
        }

        return $this->clientService->list($request);
    }

    public function create()
    {
        return Inertia::render('Admin/Clients/Form');
    }

    public function store(ClientRequest $request)
    {
        return response()->json(
            [
                'message' => 'Cliente registrado correctamente.',
                'category' => $this->clientService->create($request->validated())
            ], 201
        );
    }

    public function show(Client $client): JsonResponse
    {
        return response()->json($client);
    }

    public function edit($id)
    {
        $client = $this->clientService->getById($id);

        return Inertia::render('Admin/Clients/Form', [
            'client' => $client
        ]);
    }

    public function update(UpdateClientRequest $request, $id): JsonResponse
    {
        $client = $this->clientService->update($id, $request->validated());

        if ($client) {
            return response()->json(
                [
                    'message' => 'Cliente actualizado correctamente.',
                    'client' => $client
                ], 201
            );
        }
        
        return response()->json(['message' => 'Not found'], 404);
    }

    public function destroy($id): JsonResponse
    {
        return $this->clientService->delete($id)
            ? response()->json(['message' => 'Cliente eliminado correctamente.'])
            : response()->json(['message' => 'Not found'], 404);
    }

    public function exportPDF(Request $request)
    {
        $clients = $this->clientService->list($request);
        $pdf = Pdf::loadView('pdf.clients', ['clients' => $clients]);
        return $pdf->stream('clients.pdf');
    }

    public function exportExcel(Request $request)
    {
        $clients = $this->clientService->list($request);
        return Excel::download(new ClientsExport($clients), 'reporte-clientes.xlsx');
    }
}

