<?php

namespace App\Http\Controllers;

use App\Services\EmployeeService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Requests\{ EmployeeRequest, UpdateEmployeeRequest };
use Inertia\Inertia;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\EmployeesExport;
use Maatwebsite\Excel\Facades\Excel;

class EmployeeController extends Controller
{
    protected $employeeService;

    public function __construct(EmployeeService $employeeService)
    {
        $this->employeeService = $employeeService;
    }

    public function index()
    {
        return Inertia::render('Admin/Employees/Index');
    }

    /**
     * API employees list and search employees.
     */
    public function list(Request $request)
    {
        if ($request->query('all')) {
            return response()->json($this->employeeService->getAll()); // todos los Trabajadors
        }

        return $this->employeeService->list($request);
    }

    public function create()
    {
        return Inertia::render('Admin/Employees/Form');
    }

    public function show($id): JsonResponse
    {
        $employee = $this->employeeService->getById($id);
        return $employee 
            ? response()->json($employee) 
            : response()->json(['message' => 'Not found'], 404);
    }

    public function store(EmployeeRequest $request): JsonResponse
    {
        return response()->json(
            [
                'message' => 'Trabajador registrado correctamente.',
                'employee' => $this->employeeService->create($request->validated())
            ], 201
        );
    }

    public function edit($id)
    {
        $employee = $this->employeeService->getById($id);

        return Inertia::render('Admin/Employees/Form', [
            'employee' => $employee
        ]);
    }

    public function update(UpdateEmployeeRequest $request, $id): JsonResponse
    {
        $employee = $this->employeeService->update($id, $request->validated());

        if ($employee) {
            return response()->json(
                [
                    'message' => 'Trabajador actualizado correctamente.',
                    'employee' => $employee
                ], 201
            );
        }
        
        return response()->json(['message' => 'Not found'], 404);
    }

    public function destroy($id): JsonResponse
    {
        return $this->employeeService->delete($id)
            ? response()->json(['message' => 'Trabajador eliminado correctamente.'])
            : response()->json(['message' => 'Not found'], 404);
    }

    public function exportPDF(Request $request)
    {
        $employees = $this->employeeService->list($request);
        $pdf = Pdf::loadView('pdf.employees', ['employees' => $employees]);
        return $pdf->stream('employees.pdf');
    }

    public function exportExcel(Request $request)
    {
        $employees = $this->employeeService->list($request);
        return Excel::download(new EmployeesExport($employees), 'reporte-trabajadores.xlsx');
    }
}
