<?php
    
namespace App\Http\Controllers;

use App\Models\Product;
use App\Http\Requests\{ ProductRequest, UpdateProductRequest };
use App\Services\{ ProductService };
use Illuminate\Http\Request;
use Inertia\Inertia;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\ProductsExport;
use Maatwebsite\Excel\Facades\Excel;

class ProductController extends Controller
{
    protected $productService;

    public function __construct(ProductService $productService)
    {
        $this->productService = $productService;
    }

    public function index()
    {
        return Inertia::render('Admin/Products/Index');
    }
    
    /**
     * Api categories list and search products.
     *
     * @return \Illuminate\Http\Response
     */
    public function list(Request $request)
    {
        if ($request->query('all')) {
            return response()->json($this->productService->getAll($request)); // Retorna todos los productos
        }

        return $this->productService->list($request);
    }

    public function create()
    {
        return Inertia::render('Admin/Products/Form');
    }

    public function show($id): JsonResponse
    {
        $product = $this->productService->getById($id);
        return $product ? response()->json($product) : response()->json(['message' => 'Not found'], 404);
    }

    public function store(ProductRequest $request)
    {
        return response()->json(
            [
                'message' => 'Producto registrado correctamente.',
                'category' => $this->productService->create($request->validated())
            ], 201
        );
    }
    
    public function edit($id)
    {
        $product = $this->productService->getById($id);

        return Inertia::render('Admin/Products/Form', [
            'product' => $product
        ]);
    }

    public function update(UpdateProductRequest $request, $id)
    {
        $product = $this->productService->update($id, $request->validated());

        if ($product) {
            return response()->json(
                [
                    'message' => 'Producto actualizado correctamente.',
                    'product' => $product
                ], 201
            );
        }
        
        return response()->json(['message' => 'Not found'], 404);
    }

    public function destroy($id)
    {
        return $this->productService->delete($id)
            ? response()->json(['message' => 'Producto eliminado correctamente.'])
            : response()->json(['message' => 'Not found'], 404);
    }

    public function exportPDF(Request $request)
    {
        $products = $this->productService->list($request);
        $pdf = Pdf::loadView('pdf.products', ['products' => $products])->setPaper('a4', 'landscape');

        return $pdf->stream('products.pdf');
    }

    public function exportExcel(Request $request)
    {
        $products = $this->productService->list($request);
        return Excel::download(new ProductsExport($products), 'reporte-productos.xlsx');
    }
}
    