<?php

namespace App\Http\Controllers;

use App\Http\Requests\{ ProductCorrectionRequest };
use App\Models\Client;
use App\Services\ProductCorrectionService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\ProductCorrectionsExport;
use Maatwebsite\Excel\Facades\Excel;

class ProductCorrectionController extends Controller
{
    protected $productCorrectionService;

    public function __construct(ProductCorrectionService $productCorrectionService)
    {
        $this->productCorrectionService = $productCorrectionService;
    }

    public function index()
    {
        return Inertia::render('Admin/ProductsCorrections/Index');
    }

    /**
     * Api categories list and search products.
     *
     * @return \Illuminate\Http\Response
     */
    public function list(Request $request)
    {
        if ($request->query('all')) {
            // Retorna todos los clientes sin paginación
            return response()->json($this->productCorrectionService->getAll()); 
        }

        return $this->productCorrectionService->list($request);
    }

    public function create()
    {
        return Inertia::render('Admin/ProductsCorrections/Form');
    }

    public function store(ProductCorrectionRequest $request)
    {
        return response()->json(
            [
                'message' => 'Envío de productos registrado correctamente.',
                'category' => $this->productCorrectionService->create($request->validated())
            ], 201
        );
    }

    public function getById($id)
    {
        return response()->json(
            [
                'productCorrection' => $this->productCorrectionService->getById($id)
            ], 201
        );
    }

    public function show($id)
    {
        $productCorrection = $this->productCorrectionService->getById($id);

        return Inertia::render('Admin/ProductsCorrections/Show', [
            'productCorrection' => $productCorrection
        ]);
    }

    public function update(UpdateClientRequest $request, $id): JsonResponse
    {
        $client = $this->productCorrectionService->update($id, $request->validated());

        if ($client) {
            return response()->json(
                [
                    'message' => 'Cliente actualizado correctamente.',
                    'client' => $client
                ], 201
            );
        }
        
        return response()->json(['message' => 'Not found'], 404);
    }

    public function checkReceived($id): JsonResponse
    {
        return $this->productCorrectionService->checkReceived($id)
            ? response()->json(['message' => 'Los productos enviador fueron marcados como recibidos.'])
            : response()->json(['message' => 'Not found'], 404);
    }

    public function markAsPaid($id)
    {
        $productCorrection = $this->productCorrectionService->markAsPaid($id);

        if ($productCorrection) {
            return response()->json(
                [
                    'message' => 'Se actualizó el envío como pagado.',
                    'productCorrection' => $productCorrection
                ], 201
            );
        }
        
        return response()->json(['message' => 'Not found'], 404);
    }

    public function exportPDF(Request $request)
    {
        $productCorrections = $this->productCorrectionService->list($request);
        $totalGeneral = $productCorrections->sum('price');
        $pdf = Pdf::loadView('pdf.product-corrections', 
                    [
                        'productCorrections' => $productCorrections,
                        'totalGeneral' => $totalGeneral
                    ]);
        return $pdf->stream('purchases.pdf');
    }

    public function exportExcel(Request $request)
    {
        $purchases = $this->productCorrectionService->list($request);
        return Excel::download(new ProductCorrectionsExport($purchases), 'reporte-productos-enviados.xlsx');
    }

    public function toPayList()
    {
        $productCorrectionsList = $this->productCorrectionService->toPayList();
        return response()->json($productCorrectionsList);
    }
}
