<?php

namespace App\Http\Controllers;

use App\Http\Requests\PurchaseRequest;
use App\Models\Purchase;
use App\Services\PurchaseService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\PurchasesExport;
use Maatwebsite\Excel\Facades\Excel;

class PurchaseController extends Controller
{
    protected $purchaseService;

    public function __construct(PurchaseService $purchaseService)
    {
        $this->purchaseService = $purchaseService;
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $startDate = date('Y-m-01');
        $endDate = date('Y-m-t');

        return Inertia::render('Admin/Purchases/Index', [
            'startDate' => $startDate,
            'endDate' => $endDate
        ]);
    }

    /**
     * Api categories list and search products.
     *
     * @return \Illuminate\Http\Response
     */
    public function list(Request $request)
    {
        return $this->purchaseService->list($request);
    }

    public function create()
    {
        return Inertia::render('Admin/Purchases/Form');
    }

    public function store(PurchaseRequest $request): JsonResponse
    {
        return response()->json(
            [
                'message' => 'Compra registrada correctamente.',
                'purchase' => $this->purchaseService->create($request->validated())
            ], 201
        );
    }

    public function edit($id)
    {
        $purchase = $this->purchaseService->getById($id);

        return Inertia::render('Admin/Purchases/Form', [
            'purchase' => $purchase
        ]);
    }

    public function show($id)
    {
        $purchase = $this->purchaseService->getById($id);

        return Inertia::render('Admin/Purchases/Show', [
            'purchase' => $purchase
        ]);
    }

    public function getById($purchaseId): JsonResponse
    {
        return response()->json(
            [
                'purchase' => $this->purchaseService->getByid($purchaseId)
            ], 201
        );
    }

    public function update(PurchaseRequest $request, $purchaseId): JsonResponse
    {
        return response()->json(
            [
                'message' => 'Compra actualizada correctamente.',
                'purchase' => $this->purchaseService->update($request->validated(), $purchaseId)
            ], 201
        );
    }

    public function updateStatus(Request $request, $purchaseId): JsonResponse
    {
        return response()->json(
            [
                'message' => 'Compra actualizada correctamente.',
                'purchase' => $this->purchaseService->updateStatus($purchaseId, $request->order_status)
            ], 201
        );
    }

    public function destroy($id): JsonResponse
    {
        return $this->purchaseService->delete($id)
            ? response()->json(['message' => 'Compra cancelada correctamente.'])
            : response()->json(['message' => 'Not found'], 404);
    }

    public function toPayList()
    {
        $purchasesList = $this->purchaseService->toPayList();
        return response()->json($purchasesList);
    }

    public function exportPDF(Request $request)
    {
        $purchases = $this->purchaseService->list($request);
        $totalAdvancePayment = $purchases->sum('advance_payment');
        $totalBalance = $purchases->sum(function ($c) {
            return $c->total_amount - $c->advance_payment;
        });
        $totalGeneral = $purchases->sum('total_amount');
        $pdf = Pdf::loadView('pdf.purchases', 
                    [
                        'purchases' => $this->purchases,
                        'totalAdvancePayment' => $totalAdvancePayment,
                        'totalBalance' => $totalBalance,
                        'totalGeneral' => $totalGeneral
                    ]);
        return $pdf->stream('purchases.pdf');
    }

    public function exportPDFQuote($id)
    {
        $purchase = $this->purchaseService->getByid($id);

        $pdf = Pdf::loadView('pdf.purchases-quotes', 
                    [
                        'purchase' => $purchase,
                    ]);
        return $pdf->stream('purchases-quotes.pdf');
    }

    public function exportExcel(Request $request)
    {
        $purchases = $this->purchaseService->list($request);
        return Excel::download(new PurchasesExport($purchases), 'reporte-compras.xlsx');
    }
}
