<?php
    
namespace App\Http\Controllers;


use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Spatie\Permission\Models\Role;
use App\Services\RoleService;
use Spatie\Permission\Models\Permission;
use DB;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Inertia\Inertia;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\RolesExport;
use Maatwebsite\Excel\Facades\Excel;

class RoleController extends Controller
{
    protected $roleService;

    public function __construct(RoleService $roleService)
    {
        $this->roleService = $roleService;
    }


         //$this->middleware('permission:role-list|role-create|role-edit|role-delete', ['only' => ['index','store']]);
         //$this->middleware('permission:role-create', ['only' => ['create','store']]);
         //$this->middleware('permission:role-edit', ['only' => ['edit','update']]);
         //$this->middleware('permission:role-delete', ['only' => ['destroy']]);
    
    
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        return Inertia::render('Admin/Roles/Index');

        $roles = Role::orderBy('id','DESC')->paginate(5);
        return view('roles.index',compact('roles'))->with('i', ($request->input('page', 1) - 1) * 5);
    }
    
    /**
     * Api categories list and search products.
     *
     * @return \Illuminate\Http\Response
     */
    public function list(Request $request)
    {
        if ($request->query('all')) {
            return response()->json($this->roleService->getAll()); // Retorna todas las categorías sin paginación
        }

        return $this->roleService->list($request);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return Inertia::render('Admin/Roles/Form', [
            'permissions' => Permission::get()
        ]);
        
        return view('roles.create',compact('permission'));
    }
    
    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->validate($request,
            [
                'user_id' => 'nullable',
                'name' => 'required|unique:roles,name|min:2|max:30',
                'permissions' => 'required',
            ],
            [
                'name.required' => 'El campo es requerido',
                'name.unique' => 'El rol ya existe',
                'name.min' => 'Ingrese mínimo 2 dígitos',
                'name.max' => 'Ingrese máximo 30 dígitos',

                'permissions.required' => 'Seleccione uno o muchos permisos',
            ]
        );

        return response()->json(
            [
                'message' => 'Rol registrado correctamente.',
                'purchase' => $this->roleService->create($request)
            ], 201
        );
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id): View
    {
        $role = Role::find($id);
        $rolePermissions = Permission::join("role_has_permissions","role_has_permissions.permission_id","=","permissions.id")
            ->where("role_has_permissions.role_id",$id)
            ->get();
    
        return view('roles.show',compact('role','rolePermissions'));
    }
    
    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $role = Role::find($id);
        $permission = Permission::get();
        $rolePermissions = DB::table("role_has_permissions")->where("role_has_permissions.role_id",$id)
            ->pluck('role_has_permissions.permission_id','role_has_permissions.permission_id')
            ->all();
        
        return Inertia::render('Admin/Roles/Form', [
            'role' => Role::find($id),
            'permissions' => Permission::get(),
            'rolePermissions' => $rolePermissions
        ]);
        return view('roles.edit',compact('role','permission','rolePermissions'));
    }
    
    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $this->validate($request,
            [
                'user_id' => 'nullable',
                'name' => 'required|string|min:2|max:30|unique:roles,name,'. $id,
                'permissions' => 'required',
            ],
            [
                'name.required' => 'El campo es requerido',
                'name.unique' => 'El rol ya existe',
                'name.min' => 'Ingrese mínimo 2 dígitos',
                'name.max' => 'Ingrese máximo 30 dígitos',

                'permissions.required' => 'Seleccione uno o muchos permisos',
            ]
        );
        
        $role = $this->roleService->update($id, $request);

        if ($role) {
            return response()->json(
                [
                    'message' => 'Rol actualizado correctamente.',
                    'role' => $role
                ], 201
            );
        }
        
        return response()->json(['message' => 'Not found'], 404);
    }

    public function destroy($id)
    {
        return $this->roleService->delete($id)
            ? response()->json(['message' => 'Rol eliminado correctamente.'])
            : response()->json(['message' => 'Not found'], 404);
    }

    public function exportPDF(Request $request)
    {
        $roles = $this->roleService->list($request);

        $pdf = Pdf::loadView('pdf.roles', ['roles' => $roles]);
        return $pdf->stream('roles.pdf');
    }

    public function exportExcel(Request $request)
    {
        $roles = $this->roleService->list($request);
        return Excel::download(new RolesExport($roles), 'reporte-roles.xlsx');
    }
}
