<?php

namespace App\Http\Controllers;

use App\Http\Requests\SaleRequest;
use App\Models\{ Setting, Sale};
use App\Services\{ CashRegisterService, SaleService };
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\SalesExport;
use Maatwebsite\Excel\Facades\Excel;

use Auth;
class SaleController extends Controller
{
    protected $cashRegisterService;
    protected $saleService;

    public function __construct(CashRegisterService $cashRegisterService, SaleService $saleService)
    {
        $this->cashRegisterService = $cashRegisterService;
        $this->saleService = $saleService;
    }

    public function index()
    {
        $startDate = date('Y-m-01');
        $endDate = date('Y-m-t');

        return Inertia::render('Admin/Sales/Index', [
            'startDate' => $startDate,
            'endDate' => $endDate,
            'cashRegister' => $this->cashRegisterService->getByUserFirst(Auth::user()->id),
        ]);
    }
    
    /**
     * Api categories list and search products.
     *
     * @return \Illuminate\Http\Response
     */
    public function list(Request $request)
    {
        return $this->saleService->list($request);
    }

    public function create()
    {
        return Inertia::render('Admin/Sales/Form', [
            'cashRegister' => $this->cashRegisterService->getByUserFirst(Auth::user()->id),
        ]);
    }

    public function store(SaleRequest $request): JsonResponse
    {
        $sale = $this->saleService->create($request->validated());
        return response()->json(['message' => 'Venta registrada correctamente', 'sale' => $sale], 201);
    }

    public function show($id)
    {
        $sale = $this->saleService->getById($id);

        return Inertia::render('Admin/Sales/Show', [
            'sale' => $sale
        ]);
    }

    public function update(SaleRequest $request, Sale $sale): JsonResponse
    {
        $updatedSale = $this->saleService->updateSale($sale, $request->validated());
        return response()->json(['message' => 'Venta actualizada correctamente', 'sale' => $updatedSale]);
    }

    public function destroy($id): JsonResponse
    {
        return $this->saleService->delete($id)
            ? response()->json(['message' => 'Venta cancelada correctamente.'])
            : response()->json(['message' => 'Not found'], 404);
    }

    public function toCollectList()
    {
        $salesList = $this->saleService->toCollectList();
        return response()->json($salesList);
    }

    public function exportPDF(Request $request)
    {
        $sales = $this->saleService->list($request);
        $totalAdvancePayment = $sales->sum('advance_payment');
        $totalBalance = $sales->sum(function ($c) {
            return $c->total - $c->advance_payment;
        });
        $totalGeneral = $sales->sum('total');
        $pdf = Pdf::loadView('pdf.sales', 
                    [
                        'sales' => $sales,
                        'totalAdvancePayment' => $totalAdvancePayment,
                        'totalBalance' => $totalBalance,
                        'totalGeneral' => $totalGeneral
                    ]);
        return $pdf->stream('sales.pdf');
    }

    public function exportExcel(Request $request)
    {
        $sales = $this->saleService->list($request);
        return Excel::download(new SalesExport($sales), 'reporte-ventas.xlsx');
    }

    public function printTicket(Request $request, $id) 
    {
        $setting = Setting::first();
        $sale = Sale::with(['details.product', 'details.product_variant.color', 'details.product_variant.size', 'customer', 'branch', 'user'])->findOrFail($id);

        if($request->type == "Ticket") {
            $pdf = Pdf::loadView('pdf-tickets.sales', compact('sale', 'setting'));
            $pdf->setPaper([0, 0, 226.77, 566.93]);
        }
        else {
            $pdf = Pdf::loadView('pdf-tickets.sales_a4', compact('sale', 'setting'));
        }
        
        return $pdf->stream("ticket_venta_$id.pdf");
    }
}
