<?php

namespace App\Http\Controllers;

use App\Http\Requests\{ SupplierRequest, UpdateSupplierRequest };
use App\Models\Supplier;
use App\Services\SupplierService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\SuppliersExport;
use Maatwebsite\Excel\Facades\Excel;

class SupplierController extends Controller
{
    protected $supplierService;

    public function __construct(SupplierService $supplierService)
    {
        $this->supplierService = $supplierService;
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return Inertia::render('Admin/Suppliers/Index');
    }

    /**
     * Api categories list and search products.
     *
     * @return \Illuminate\Http\Response
     */
    public function list(Request $request)
    {
        if ($request->query('all')) {
            return response()->json($this->supplierService->getAll($request)); // Retorna todas las categorías sin paginación
        }

        return $this->supplierService->list($request);
    }

    public function create()
    {
        return Inertia::render('Admin/Suppliers/Form');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(SupplierRequest $request)
    {
        return response()->json(
            [
                'message' => 'Proveedor registrado correctamente.',
                'category' => $this->supplierService->create($request->validated())
            ], 201
        );
    }

    public function show(Supplier $supplier): JsonResponse
    {
        return response()->json($supplier);
    }

    public function edit($id)
    {
        $supplier = $this->supplierService->getById($id);

        return Inertia::render('Admin/Suppliers/Form', [
            'supplier' => $supplier
        ]);
    }

    public function update(UpdateSupplierRequest $request, $id): JsonResponse
    {
        $supplier = $this->supplierService->update($id, $request->validated());

        if ($supplier) {
            return response()->json(
                [
                    'message' => 'Proveedor actualizado correctamente.',
                    'supplier' => $supplier
                ], 201
            );
        }
        
        return response()->json(['message' => 'Not found'], 404);
    }

    public function destroy($id): JsonResponse
    {
        return $this->supplierService->delete($id)
            ? response()->json(['message' => 'Proveedor eliminado correctamente.'])
            : response()->json(['message' => 'Not found'], 404);
    }

    public function exportPDF(Request $request)
    {
        $suppliers = $this->supplierService->list($request);

        $pdf = Pdf::loadView('pdf.suppliers', ['suppliers' => $suppliers]);
        return $pdf->stream('suppliers.pdf');
    }

    public function exportExcel(Request $request)
    {
        $suppliers = $this->supplierService->list($request);
        return Excel::download(new SuppliersExport($suppliers), 'reporte-proveedores.xlsx');
    }
}
