<?php
    
namespace App\Http\Controllers;
    
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\User;
use App\Http\Requests\{ SupplierRequest, UpdateUserRequest, UserChangePasswordRequest };
use Spatie\Permission\Models\Role;
use DB;
use Hash;
use Mail;
use Session;
use Illuminate\Support\Arr;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use App\Services\UserService;
use Inertia\Inertia;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\UsersExport;
use Maatwebsite\Excel\Facades\Excel;

class UserController extends Controller
{
    protected $userService;

    public function __construct(UserService $userService)
    {
        $this->userService = $userService;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        return Inertia::render('Admin/Users/Index');

        $data = User::latest()->paginate(5);
        
        //return Inertia::render('Admin/Users/Index');

        return view('users.index',compact('data'))
            ->with('i', ($request->input('page', 1) - 1) * 5);
    }
    
    /**
     * Api categories list and search products.
     *
     * @return \Illuminate\Http\Response
     */
    public function list(Request $request)
    {
        return $this->userService->list($request);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $roles = Role::pluck('name','name')->all();

        return Inertia::render('Admin/Users/Form', [
            'roles' => $roles
        ]);

        return view('users.create',compact('roles'));
        return Inertia::render('Admin/Users/Form');
    }
    
    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request['confirm-password'] = $request['confirmPassword'];

        $this->validate($request, [
            'name' => 'required',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|same:confirm-password',
            'roles' => 'required'
        ]);
    
        $input = $request->all();
        $input['password'] = Hash::make($input['password']);
    
        $user = User::create($input);
        $user->assignRole($request->input('roles'));
        
        return response()->json(
            [
                'message' => 'Usuario registrado correctamente.'
                //'user' => $this->supplierService->create($request->validated())
            ], 201
        );
    }
    
    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id): View
    {
        $user = User::find($id);

        return view('users.show',compact('user'));
    }
    
    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = $this->userService->getById($id);
        $roles = Role::pluck('name','name')->all();
        $roleName = count($user->roles) > 0 ? $user->roles[0]->name : '';

        return Inertia::render('Admin/Users/Form', [
            'roles' => $roles,
            'user' => $user,
            'roleName' => $roleName,
        ]);
    }
    
    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateUserRequest $request, $id)
    {
        $user = $this->userService->update($id, $request->validated());

        if ($user) {
            return response()->json(
                [
                    'message' => 'Usuario actualizado correctamente.',
                ], 201
            );
        }
        
        return response()->json(['message' => 'Not found'], 404);
    }
    
    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        return $this->userService->delete($id)
            ? response()->json(['message' => 'Usuario eliminado correctamente.'])
            : response()->json(['message' => 'Not found'], 404);
    }

    public function viewChangePassword(Request $request)
    {
        return Inertia::render('Admin/Users/ChangePassword');
    }

    public function changePassword(UserChangePasswordRequest $request, $id)
    {
        $user = $this->userService->changePassword($request, $id);

        return response()->json(['message' => 'Tu contraseña fue actualizada correctamente.']);
    }

    public function passwordReset(Request $request)
    {
        $user = $this->userService->getByEmail($request->email);

        if ($user) {
            $rememberToken = $this->userService->updateNewRememberToken($user->id);
        
            Mail::send('auth.passwords.email_reset_password', ['user' => $user, 'rememberToken' => $rememberToken], function($msj) use($user){
                $msj->from(env('MAIL_USERNAME'), env('MAIL_FROM_NAME'));
                $msj->subject('¡Restablecer contraseña!');
                $msj->to("wilcm123@gmail.com");
            });

            Session::flash('success', 'Te enviamos un mensaje a tu correo electrónico para que puedas reestablecer tu contraseña.');
            return redirect('password-reset');
        }
        else {
            Session::flash('error', 'El correo electrónico ingresado no existe.');
            return redirect('password-reset');
        }
    }

    public function ConfirmationPassword(Request $request)
    {
        if ($_GET) {
            $user = $this->userService->getByRememberToken($request->token);

            if ($user) {
                return view('auth/passwords/confirm', compact('user'));
            }
            
            Session::flash('warning', 'El token para actualizar su password ya fue usado, puedes solicitar uno nuevo en este formulario.');
            return redirect('password-reset');
        }
        
        if ($this->userService->validatePasswords($request)) {
            $user = $this->userService->updatePassword($request);

            Session::flash('success', 'Tu contraeña fue actualizada correctamente.');
            return redirect('password-reset');
        }

        Session::flash('error', 'Ambas contraseñas deben ser iguales.');
        return redirect('password-reset/confirmation?token='. $request->token);
    }

    public function exportPDF(Request $request)
    {
        $users = $this->userService->list($request);

        $pdf = Pdf::loadView('pdf.users', ['users' => $users]);
        return $pdf->stream('users.pdf');
    }

    public function exportExcel(Request $request)
    {
        $users = $this->userService->list($request);
        return Excel::download(new UsersExport($users), 'reporte-usuarios.xlsx');
    }
}
