<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class ProductVariantRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true; // Cambia si deseas aplicar políticas de acceso
    }
 
    public function rules(): array
    {
        return [
            'user_id' => "required",
            'product_id' => 'required|exists:products,id',
            'size_id'       => 'required|min:1',
            'color_id'      => 'required|min:1',
            'purchase_price' => 'required|numeric|min:0', 
            'price'      => 'required|numeric|min:0',
            'min_price' => 'nullable|numeric|min:0',
            'stock'      => 'required|integer|min:0',
            'min_stock' => 'required|integer|min:0',
            'main_image' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:5120',
            'supplier_barcode' => 'nullable|string|min:0|max:100',
        ];
    }

    public function withValidator($validator)
    {
        $validator->after(function ($validator) {
            if ($this->variantExists()) {
                $validator->errors()->add('size_id', 'Ya existe una variante con esa talla y color.');
            }
        });
    }

    public function messages(): array
    {
        return [
            'product_id.required' => 'El producto es obligatorio.',
            'product_id.exists'   => 'El producto seleccionado no existe.',
            
            'size_id.required' => 'Seleccione una talla.',
            'size_id.min' => 'Ingrese mínimo 1 dígitos.',

            'color_id.required'      => 'Seleccione un color.',
            'color_id.min'            => 'Ingrese el valor de 1.',
            
            'purchase_price.required'      => 'El campo es obligatorio.',
            'purchase_price.numeric'       => 'El precio debe ser numérico.',
            'purchase_price.min'           => 'El precio debe ser mayor o igual a 0.',

            'price.required'      => 'El campo es obligatorio.',
            'price.numeric'       => 'El precio debe ser numérico.',
            'price.min'           => 'El precio debe ser mayor o igual a 0.',

            'min_price.numeric' => 'Ingrese un valor numérico',
            'min_price.min' => 'Ingrese mínimo el valor de 0',
            
            'main_image.required'      => 'El campo es obligatorio.',
            'main_image.image' => 'Seleccione una imagen',
            'main_image.mimes' => 'Seleccione una imagen con el formato: jpeg,png,jpg,gif',
            'main_image.max' => 'La imagen debe pesar máximo 2mb',

            'stock.required'      => 'El campo es obligatorio.',
            'stock.integer'       => 'El stock debe ser un número entero.',
            'stock.min'           => 'El stock debe ser mayor o igual a 0.',

            'min_stock.required'      => 'El campo es obligatorio.',
            'min_stock.integer'       => 'El debe ser un número entero.',
            'min_stock.min'           => 'El debe ser mayor o igual a 0.',

            'supplier_barcode.max' => 'La descripción debe tener máximo 100 dígitos',
        ];
    }

    protected function variantExists(): bool
    {
        $query = \App\Models\ProductVariant::where('product_id', $this->product_id)
            ->where('size_id', $this->size_id)
            ->where('color_id', $this->color_id );

        // Si es actualización, ignorar el ID actual
        if ($this->route('product_variant')) {
            $query->where('id', '!=', $this->route('product_variant')->id);
        }

        return $query->exists();
    }
}
