<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class ProformaRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true; // Aquí puedes implementar lógica de permisos si quieres
    }

    public function rules(): array
    {
        return [
            'customer_id' => 'required|exists:clients,id',
            'status' => 'required|in:Pedido,Atendido,Cancelado,Confirmado',
            'total' => 'required|numeric|min:0',
            'has_furniture_service' => 'required|min:0|max:1',
            'delivery_date' => 'required|date',

            'items' => 'required|array|min:1',

            // Reglas para productos de referencia
            'items.*.product_id' => 'nullable',
            'items.*.product_variant_id' => 'nullable',

            // Reglas para personalizados (si product_id es null)
            'items.*.custom_name'        => 'nullable|string|min:3|max:100',
            'items.*.custom_description' => 'nullable|string|max:500',
            'items.*.custom_measurements'=> 'nullable|string|max:100',

            // Cantidad y precio (aplican siempre)
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.price' => 'required|numeric|min:0',
        ];
    }

    public function withValidator($validator)
    {
        $validator->after(function ($validator) {
            if ($this->items && is_array($this->items)) {
                foreach ($this->items as $index => $item) {
                    if (empty($item['product_id']) && empty($item['custom_name'])) {
                        $validator->errors()->add(
                            "items.$index.custom_name",
                            "El nombre personalizado es obligatorio si no seleccionas un producto."
                        );
                    }
                }
            }
        });
    }

    public function messages(): array
    {
        return [
            'customer_id.required' => 'El cliente es obligatorio',
            'customer_id.exists' => 'El cliente seleccionado no existe',

            'status.required' => 'El estado es obligatorio',
            'status.in' => 'El estado debe ser "nuevo" o "atendido"',

            'total.required' => 'El campo es obligatorio',
            'total.numeric' => 'Ingrese un valor numérico',

            'has_furniture_service.required' => 'El campo es obligatorio',
            'has_furniture_service.min' => 'Ingrese mínimo 0',
            'has_furniture_service.max' => 'Ingrese máximo 1',

            'items.required' => 'Debe agregar al menos un producto',
            'items.array' => 'El detalle debe ser un array',
            'items.min' => 'Debe registrar al menos un producto en la proforma',

            'items.*.custom_name.required_without' => 'El nombre del producto personalizado es obligatorio cuando no se selecciona un producto existente',
            'items.*.custom_name.min' => 'El nombre del producto personalizado debe tener al menos 3 caracteres',
            'items.*.custom_name.max' => 'El nombre del producto personalizado no debe superar los 100 caracteres',

            'items.*.quantity.required' => 'La cantidad es obligatoria',
            'items.*.quantity.min' => 'La cantidad debe ser al menos 1',

            'items.*.price.required' => 'El precio es obligatorio',
            'items.*.price.min' => 'El precio no puede ser negativo',
        ];
    }
}
