<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class UpdateProductRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [
            'user_id' => "required",
            'brand_id' => 'required|exists:brands,id',
            'category_id' => 'required|exists:categories,id',
            'name' => [
                'required',
                'string',
                'min:3',
                'max:100',
                Rule::unique('products')->where(function ($query) {
                    return $query->where('brand_id', $this->brand_id);
                })->ignore($this->id)
            ],
            'supplier_barcode' => 'nullable|string|min:0|max:100',
            'description' => 'nullable|string|min:3|max:300',
            'purchase_price' => 'nullable|numeric|min:0',
            'price' => 'nullable|numeric|min:0',
            'min_price' => 'nullable|numeric|min:0',
            'stock' => 'nullable|integer|min:0',
            'min_stock' => 'nullable|integer|min:0',
            'main_image' => 'nullable|mimes:jpeg,png,jpg,gif,webp|max:5120',
            'has_variants' => 'required|numeric|min:0|max:1',
        ];
    }

    public function messages(): array
    {
        return [
            'brand_id.required' => 'Seleccione una marca',
            'brand_id.exists' => 'La marca no existe',

            'category_id.required' => 'Seleccione una categoría',
            'category_id.exists' => 'La categoría no existe',

            'name.required' => 'El campo es obligatorio',
            'name.unique' => 'Ya existe un producto con este nombre en la misma marca',
            'name.min' => 'El nombre debe tener mínimo 3 caracteres',
            'name.max' => 'El nombre debe tener máximo 100 caracteres',

            'supplier_barcode.max' => 'La descripción debe tener máximo 100 dígitos',
            
            'description.min' => 'La descripción debe tener mínimo 3 caracteres',
            'description.max' => 'La descripción debe tener máximo 300 caracteres',

            'purchase_price.numeric' => 'Ingrese un valor numérico',
            'purchase_price.min' => 'Ingrese mínimo el valor de 0',

            'price.numeric' => 'Ingrese un valor numérico',
            'price.min' => 'Ingrese mínimo el valor de 0',

            'min_price.numeric' => 'Ingrese un valor numérico',
            'min_price.min' => 'Ingrese mínimo el valor de 0',
            
            'stock.integer' => 'Ingrese un valor numérico entero',
            'stock.min' => 'Ingrese mínimo el valor de 0',

            'min_stock.integer' => 'Ingrese un valor numérico entero',
            'min_stock.min' => 'Ingrese mínimo el valor de 0',

            'main_image.mimes' => 'Seleccione una imagen con el formato: jpeg,png,jpg,gif,webp',
            'main_image.max' => 'La imagen debe pesar máximo 5mb',

            'has_variants.required' => 'Seleccione una opción',
            'has_variants.numeric' => 'Ingrese un valor numérico',
            'has_variants.min' => 'Ingrese mínimo el valor de 0',
            'has_variants.max' => 'Ingrese máximo el valor de 1',
        ];
    }
}
