<?php

namespace App\Services;

use App\Services\{ ProductService };
use App\Models\{ Product, BranchProductStock };
use App\Models\StockAdjustment;
use Illuminate\Database\Eloquent\Collection;

use DB;

class BranchProductStockService
{ 
    protected $productService;

    public function __construct(ProductService $productService)
    {
        $this->productService = $productService;
    }

    public function list($request)
    {
        $branchId = $request->get('branch_id');
        $search = $request->get('searchQuery');

        if (!$branchId) {
            return response()->json([], 400);
        }

        return response()->json(
            $this->searchByBranch($branchId, $search)
        );
    }

    public function getById($id)
    {
        return BranchProductStock::find($id);
    }

    public function searchByBranch($branchId, $search = null)
    {
        return Product::with(['variants', 'variants.size', 'variants.color', 
                            'branchStocks' => function($query) use($branchId) {
                                $query->where('branch_id', $branchId);
                            }
                        ])
                        ->whereHas('branchStocks', function($q) use ($branchId) {
                            $q->where('branch_id', $branchId)
                              ->where('stock', '>', 0);
                        })
                        ->when($search, function($q) use ($search) {
                            $q->where('name', 'like', "%$search%");
                        })
                        ->get();
    }

    public function listByProduct($productId)
    {
        return BranchProductStock::with('branch')->where('product_id', $productId)->get();
    }

    public function updateStock($id, $stock, $userId)
    {
        return DB::transaction(function () use ($id, $stock, $userId) {
            $branchProductStock = BranchProductStock::find($id);

            StockAdjustment::create([
                'branch_product_stock_id' => $branchProductStock->id,
                'product_id'        => $branchProductStock->product_id,
                'product_variant_id'=> $branchProductStock->product_variant_id, // si existe
                'branch_id'         => $branchProductStock->branch_id,
                'user_id'           => $userId,
                'old_stock'         => $branchProductStock->stock,
                'new_stock'         => $stock,
                'note'              => 'Ajuste manual por super admin'
            ]);
            
            $branchProductStock->stock = $stock;

            return $branchProductStock->save();
        });
    }

    public function updateStockPurchase($branchId, $productId, $productVariantId = null, $quantity)
    {
        if ($productVariantId) {
            $branchProductStock = BranchProductStock::where('branch_id', $branchId)
                                                ->where('product_id', $productId)
                                                ->where('product_variant_id', $productVariantId)
                                                ->first();
        }
        else {
            $branchProductStock = BranchProductStock::where('branch_id', $branchId)
                                                ->where('product_id', $productId)
                                                ->first();
        }
        
        $branchProductStock->stock = $branchProductStock->stock + $quantity;
        $branchProductStock->save();
    }

    public function updateStockById($branchProductStockId, $quantity)
    {
        $branchProductStock = BranchProductStock::find($branchProductStockId);
        $branchProductStock->stock = $branchProductStock->stock + $quantity;
        $branchProductStock->save();
    }
}
