<?php

namespace App\Services;

use App\Models\MedicalConsultation;
use App\Services\ProductService;
use Illuminate\Support\Facades\DB;

class MedicalConsultationService 
{
    protected $productService;

    public function __construct(ProductService $productService)
    {
        $this->productService = $productService;
    }

    public function list($request)
    {
        $medicalConsultations = MedicalConsultation::with('customer')->where('customer_id', $request->clientId);

        if ($request->start_date && $request->end_date) {
            $startDate = $request->start_date .' 00:00:00';
            $endDate = $request->end_date .' 23:59:59';

            $medicalConsultations = $medicalConsultations->whereBetween('created_at', [$startDate, $endDate]);
        }

        if ($request->type) {
            $medicalConsultations = $medicalConsultations->where('type', $request->type);
        }

        return $medicalConsultations->orderBy('created_at', 'desc')->paginate(20);
    }

    public function create(array $data)
    {
        $data['invoice_number'] = $this->generateCorrelation();
        $data['consultation_date'] = date('Y-m-d');

        return MedicalConsultation::create($data);
    }

    public function update(Sale $sale, array $data)
    {
        return DB::transaction(function () use ($sale, $data) {
            // Update sale
            $sale->update([
                'customer_id' => $data['customer_id'],
                'sale_date' => $data['sale_date'],
                'total' => $data['total'],
            ]);

            // Delete old details and insert new ones
            $sale->saleDetails()->delete();

            foreach ($data['saleDetails'] as $detail) {
                SaleDetail::create([
                    'sale_id' => $sale->id,
                    'product_id' => $detail['product_id'],
                    'quantity' => $detail['quantity'],
                    'price' => $detail['price'],
                    'subtotal' => $detail['quantity'] * $detail['price'],
                ]);
            }

            return $sale;
        });
    }

    public function delete(int $id): bool
    {
        return DB::transaction(function () use ($id) {
            $sale = Sale::find($id);
            $sale->status = 'canceled';
            $sale->canceled_date = date('Y-m-d H:i:s');

            $saleDetails =  SaleDetail::where('sale_id', $id)->get();

            foreach ($saleDetails as $detail) {
                $this->productService->updateStockNotVariant($detail->product_id, $detail->quantity, $detail->unit_price);
            }

            return $sale->save() ? true : false;
        });
    }

    public function generateCorrelation()
    {
        $lastInvoice = MedicalConsultation::orderBy('id', 'desc')->first();
        $lastNumber = $lastInvoice ? intval($lastInvoice->invoice_number) : 0;
        $nextNumber = str_pad($lastNumber + 1, 6, '0', STR_PAD_LEFT);

        return $nextNumber;
    }
}
