<?php

namespace App\Services;

use App\Models\{ ProductCorrection, ProductCorrectionItem };
use Illuminate\Database\Eloquent\Collection;
use App\Services\SaleDetailService;

use DB;

class ProductCorrectionService
{
    protected $saleDetailService;

    public function __construct(SaleDetailService $saleDetailService)
    {
        $this->saleDetailService = $saleDetailService;
    }
    
    public function list($request)
    {
        $productsCorrections = ProductCorrection::with('supplier');

        if ($request->start_date && $request->end_date) {
            $startDate = $request->start_date .' 00:00:00';
            $endDate = $request->end_date .' 23:59:59';

            $productsCorrections = $productsCorrections->whereBetween('created_at', [$startDate, $endDate]);
        }

        if ($request->supplier_id) {
            $productsCorrections = $productsCorrections->where('supplier_id', $request->supplier_id);
        }

        if ($request->status) {
            $productsCorrections = $productsCorrections->where('status', $request->status);
        }

        if ($request->payment_status) {
            $productsCorrections = $productsCorrections->where('payment_status', $request->payment_status);
        }

        return $productsCorrections->orderBy('created_at', 'desc')->paginate(20);
    }

    public function getAll(): Collection
    {
        return Client::all();
    }
 
    public function getById(int $id): ?ProductCorrection
    {
        return ProductCorrection::with(['supplier', 'items.sale_detail.sale.customer'])->find($id);
    }

    public function create(array $data): ProductCorrection
    {
        return DB::transaction(function () use ($data) {
            $data['correction_date'] = date('Y-m-d');
            $data['payment_status'] = 'pending';
            $data['status'] = 'sent';

            $productCorrection = ProductCorrection::create($data);

            foreach ($data['items'] as $detail) {
                ProductCorrectionItem::create([
                    'product_correction_id' => $productCorrection->id,
                    'sale_detail_id' => $detail['id'],
                    'quantity' => $detail['quantity'],
                    'reason' => "reason",
                ]);
                
                $this->saleDetailService->updateStatusToSent($detail['id']);
            }

            return $productCorrection;
        });
    }

    public function update($id, $data): ?ProductCorrection
    {
        $ProductCorrection = ProductCorrection::find($id);
        if ($client) {
            $client->update($data);
        }

        return $client;
    }

    public function checkReceived(int $id)
    {
        $productCorrection = ProductCorrection::with('items')->find($id);

        $markAsReceived = true;

        foreach ($productCorrection->items as $productCorrectionItem) {
            if($productCorrectionItem->status != 'received')
                $markAsReceived = false;
        }

        if ($markAsReceived) {
            $productCorrection->received_date = date('Y-m-d H:i:s');
            $productCorrection->status = 'received';
            $productCorrection->save();
        }
    }

    public function markAsPaid(int $id)
    {
        $productCorrection = ProductCorrection::find($id);
        $productCorrection->payment_status = 'paid';
        
        return $productCorrection->save();
    }

    public function toPayList()
    {
        return ProductCorrection::with('supplier')->where('payment_status', 'pending')->get();
    }
}
