<?php

namespace App\Services;

use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\BranchProductStock;
use Illuminate\Support\Facades\Storage;

use DB;

class ProductService
{
    public function list($request)
    {
        if ($request->action && $request->action == 'export') {
            return Product::with(['brand', 'category', 'variants.color', 'variants.size', 'branchStocks.branch'])
                        ->orderBy('name', 'asc') 
                        ->where('name', 'like', '%'. $request->search .'%')
                        ->get();
        }

        return Product::with(['brand', 'category', 'variants.color', 'variants.size', 'variants.branch_stocks', 'variants.purchases_detail', 'variants.sales_detail', 'variants.product_tranfer_items', 'branchStocks.branch', 'sale_details', 'purchase_details', 'transfer_items'])
                        ->orderBy('name', 'asc') 
                        ->where('name', 'like', '%'. $request->search .'%')
                        ->paginate(20);
    }

    public function getAll($request)
    {
        $productId = "";
        $productIdVariant = "";

        if ($request->supplier_barcode) {
            $product = Product::where('supplier_barcode', $request->supplier_barcode)->first();

            if ($product) {
                $productId = $product->id;
            }
            else {
                $productVariant = ProductVariant::where('supplier_barcode', $request->supplier_barcode)->first();

                if ($productVariant) {
                    $productIdVariant = $productVariant->id;
                }
            }
        }

        if($productId != '' || $productIdVariant != ''){
            if ($productId != '') {
                return Product::with(['category', 'variants.color', 'variants.size', 'branchStocks.branch'])
                            ->where('id', $productId)
                            ->get();
            }
            else if ($productIdVariant != ''){
                return Product::with(['category', 'variants.color', 'variants.size', 'branchStocks.branch'])
                            ->whereHas('variants', function($q) use ($productIdVariant) {
                                $q->where('id', $productIdVariant);
                            })
                            ->get();
            }
        }
        else {
            $products = Product::with(['category', 'variants.color', 'variants.size', 'branchStocks.branch']);

            if($request->searchQuery){
                $products = $products->where('name', 'like', '%'. $request->searchQuery .'%');

                return $products->get();
            }

            return [];
        }
    }

    public function getById(int $id): ?Product
    {
        return Product::find($id);
    }

    public function create(array $data)
    {
        return DB::transaction(function () use ($data) {
            // Guardando la imagen principal
            if (isset($data['main_image'])) {
                $image     = $data['main_image'];
                $nameImage = date('YmdHis') .'.'. $image->getClientOriginalExtension();
                $image->move(public_path('img/products'), $nameImage);

                $data['main_image'] = $nameImage;
            }

            $product = Product::create([
                'user_id' => $data['user_id'],
                'category_id' => $data['category_id'],
                'brand_id' => $data['brand_id'],
                'name' => $data['name'],
                'supplier_barcode' => $data['has_variants'] ? "" : $data['supplier_barcode'],
                'description' => $data['description'],
                'type' => 'venta',
                'has_variants' => $data['has_variants'],
                'purchase_price' => $data['has_variants'] ? null : $data['purchase_price'],
                'price' => $data['has_variants'] ? null : $data['price'],
                'min_price' => $data['has_variants'] ? null : $data['min_price'],
                'stock' => $data['has_variants'] ? null : $data['stock'],
                'min_stock' => $data['min_stock'] ? $data['min_stock'] : 0,
                'main_image' => $data['main_image'],
            ]);
    
            if ($data['has_variants'] == "0") {
                BranchProductStock::create([
                    'branch_id' => 1,
                    'product_id' => $product->id,
                    'stock' => $data['stock'],
                ]);
            }
            
            return $product;
        });
    }

    public function update($id, array $data)
    {
        $product = Product::find($id);

        if (isset($data['main_image'])) {
            \File::delete(public_path('img/products/'. $product->main_image));

            $image     = $data['main_image'];
            $nameImage = date('YmdHis') .'.'. $image->getClientOriginalExtension();
            $image->move(public_path('img/products'), $nameImage);

            $data['main_image'] = $nameImage;
        } else {
            unset($data['main_image']);
        }

        $data["supplier_barcode"] = $data['has_variants'] || $data['supplier_barcode'] == "null" ? "" : $data['supplier_barcode'];
        $data['price'] = $data['has_variants'] ? null : $data['price'];
        $data['min_stock'] = $data['min_stock'] ? $data['min_stock'] : 0;

        // Actualizar producto
        $product->update($data);

        return $product;
    }

    public function delete($id)
    {
        $product = $this->getById($id);

        \File::delete(public_path('img/products/'. $product->main_image));

        return $product->delete();
    }

    public function updateStock($purchaseDetail, $type)
    {
        $product = Product::find($purchaseDetail["productId"]);
        
        if ($type == "purchase") { 
            $product->purchase_price = $purchaseDetail["purchase_price"];
            $product->price = $purchaseDetail["price"];
        }
        else {
            $product->stock = $product->stock - $purchaseDetail["quantity"];
        }
        
        $product->save();
    }
}
