<?php

// app/Services/ProductTransferService.php
namespace App\Services;

use App\Models\{ BranchProductStock, ProductTransfer, ProductTransferItem };
use Illuminate\Database\Eloquent\Collection;

use DB;

class ProductTransferService
{
    public function list($request)
    {
        return ProductTransfer::with(['fromBranch', 'toBranch', 'details.product'])
            ->orderBy('id', 'desc')
            ->paginate(20);
    }

    public function getById(int $id): ?ProductTransfer
    {
        return ProductTransfer::with(['details.product'])->find($id);
    }

    public function validateStock($fromBranchId, $details)
    {
        foreach ($details as $detail) {
            $branchProductStock = BranchProductStock::where('branch_id', $fromBranchId)
                                            ->where('product_id', $detail['productId'])
                                            ->where('product_variant_id', ($detail['variantId'] ? $detail['variantId'] : null))
                                            ->first();

            if ($branchProductStock) {
                if ($detail['quantity'] > $branchProductStock->stock) {
                    throw new \Illuminate\Validation\ValidationException(
                        validator: validator([], []),
                        response: response()->json([
                            'message' => "El producto '{$detail['name']} {$detail['size']} {$detail['color']}' no tiene suficiente stock. Disponible: {$branchProductStock->stock}"
                        ], 422)
                    );
                }
            }
            else {
                throw new \Illuminate\Validation\ValidationException(
                    validator: validator([], []),
                    response: response()->json([
                        'message' => "El producto '{$detail['name']} {$detail['size']} {$detail['color']}' no existe"
                    ], 500)
                );
            }
        }
    }

    public function create(array $data): ProductTransfer
    {
        $this->validateStock(
            $data["from_branch_id"],
            $data["details"]
        );

        return DB::transaction(function () use ($data) {
            $data["status"] = "pendiente";
            $data["pending_at"] = date('Y-m-d H:i:s');
            
            $transfer = ProductTransfer::create($data);

            foreach ($data['details'] as $detail) {
                ProductTransferItem::create([
                    'product_transfer_id' => $transfer->id,
                    'product_id' => $detail['productId'],
                    'variant_id' => $detail['variantId'],
                    'product_name' => $detail['name'],
                    'size' => $detail['size'],
                    'color' => $detail['color'],
                    'quantity' => $detail['quantity'],
                ]);
            }

            return $transfer->load('details.product');
        });
    }

    public function update($id, $data)
    {
        return DB::transaction(function () use ($id, $data) {
            $transfer = ProductTransfer::with('details')->find($id);

            if ($transfer->status == "en_transito" && $data["status"] != "completado") {
                $this->replenishStock($transfer);
            }
            
            if ($data["status"] != "completado") {
                $this->validateStock(
                    $data["from_branch_id"],
                    $data["details"]
                );
            }
            
            if (!$transfer) return null;

            // Bloquear cambio de from_branch_id
            $data['from_branch_id'] = $transfer->from_branch_id;

            if ($data["status"] == "pendiente") {
                $data["in_transit_at"] = date('Y-m-d H:i:s');
            }
            else if ($data["status"] == "en_transito") {
                $data["in_transit_at"] = date('Y-m-d H:i:s');
            }
            else if ($data["status"] == "completado") {
                $data["completed_at"] = date('Y-m-d H:i:s');
            }
            else if ($data["status"] == "cancelado") {
                $data["canceled_at"] = date('Y-m-d H:i:s');
            }

            $transfer->update([
                'to_branch_id' => $data['to_branch_id'],
                'status' => $data["status"],
                'in_transit_at' => isset($data["in_transit_at"]) ? $data["in_transit_at"] : null,
                'completed_at' => isset($data["completed_at"]) ? $data["completed_at"] : null,
                'canceled_at' => isset($data["canceled_at"]) ? $data["canceled_at"] : null,
            ]);

            $transfer->details()->delete();

            foreach ($data['details'] as $detail) {
                ProductTransferItem::create([
                    'product_transfer_id' => $transfer->id,
                    'product_id' => $detail['productId'],
                    'variant_id' => $detail['variantId'],
                    'product_name' => $detail['name'],
                    'size' => $detail['size'],
                    'color' => $detail['color'],
                    'quantity' => $detail['quantity'],
                ]);

                if ($data["status"] == "en_transito") {
                    $this->updateStockFromBrach($transfer->from_branch_id, $detail);
                }

                if ($data["status"] == "completado") {
                    $this->updateStockToBrach($transfer->to_branch_id, $detail);
                }
            }

            return $transfer;
        });
    }

    public function replenishStock($transfer)
    {
        $productTransferItemList = ProductTransferItem::where('product_transfer_id', $transfer->id)->get();

        foreach ($productTransferItemList as $detail) {
            $branchProductStock = BranchProductStock::where('branch_id', $transfer->from_branch_id)
                                                ->where('product_id', $detail->product_id)
                                                ->where('product_variant_id', $detail->variant_id)
                                                ->first();
            $branchProductStock->stock = $branchProductStock->stock + $detail->quantity;
            $branchProductStock->save();
        }
    }

    public function updateStockFromBrach($fromBranchId, $detail)
    {
        $branchProductStock = BranchProductStock::where('branch_id', $fromBranchId)
                                                ->where('product_id', $detail['productId'])
                                                ->where('product_variant_id', ($detail['variantId'] ? $detail['variantId'] : null))
                                                ->first();

        $branchProductStock->stock = $branchProductStock->stock - $detail['quantity'];
        $branchProductStock->save();
    }

    public function updateStockToBrach($toBranchId, $detail)
    {
        $branchProductStock = BranchProductStock::where('branch_id', $toBranchId)
                                                ->where('product_id', $detail['productId'])
                                                ->where('product_variant_id', ($detail['variantId'] ? $detail['variantId'] : null))
                                                ->first();

        if($branchProductStock){
            $branchProductStock->stock = $branchProductStock->stock + $detail['quantity'];
            $branchProductStock->save();
        }
        else{
            BranchProductStock::create([
                'branch_id' => $toBranchId,
                'product_id' => $detail['productId'],
                'product_variant_id' => $detail['variantId'],
                'stock' => $detail['quantity'],
            ]);
        }
    }

    public function delete(int $id): bool
    {
        $transfer = ProductTransfer::find($id);
        return $transfer ? $transfer->delete() : false;
    }
}
