<?php

namespace App\Services;

use App\Models\Proforma;
use App\Models\ProformaItem;
use App\Models\Sale;
use App\Models\User;
use App\Models\Product;
use Illuminate\Support\Facades\DB;

class ProformaService
{
    protected $branchProductStockService;

    public function __construct(BranchProductStockService $branchProductStockService)
    {   
        $this->branchProductStockService = $branchProductStockService;
    }

    public function list($request)
    {
        $proformas = Proforma::with(['customer', 'furnitures', 'items.product.branchStocks.branch']);

        if ($request->start_date && $request->end_date) {
            $startDate = $request->start_date .' 00:00:00';
            $endDate = $request->end_date .' 23:59:59';

            $proformas = $proformas->whereBetween('created_at', [$startDate, $endDate]);
        }

        if ($request->clientData) {
            $proformas = $proformas->whereHas('customer', function($query) use($request){
                $query->select("id", "first_name", "last_name", DB::Raw("CONCAT(first_name, ' ', last_name, ' ', middle_name) AS fullName"))->having("fullName", "like", "%". $request->clientData ."%");
            });
        }

        if ($request->status) {
            $proformas = $proformas->where('status', $request->status);
        }

        if ($request->action && $request->action == 'export') {
            return $proformas->orderBy('created_at', 'desc')->get();
        }

        return $proformas->orderBy('created_at', 'desc')->paginate(20);
    }

    public function getById(int $id): ?Proforma
    {
        return Proforma::with(['items.branch', 'customer'])->find($id);
    }

    public function create(array $data, array $items): Proforma
    {
        $data['items'] = $items;

        return DB::transaction(function () use ($data) {
            $proforma = Proforma::create([
                'user_id' => $data['user_id'],
                'customer_id' => $data['customer_id'],
                'status' => $data['status'] ?? 'nuevo',
                'observations' => $data['observations'] ?? null,
                'total' => $data['total'] ?? null,
                'delivery_date' => $data['delivery_date'],
                'has_furniture_service' => $data['has_furniture_service'],
            ]);

            foreach ($data['items'] as $item) {
                $nameImage = "";

                if (isset($item['custom_image'])) {
                    $image     = $item['custom_image'];
                    $nameImage = date('YmdHis') .'.'. $image->getClientOriginalExtension();
                    $image->move(public_path('img/proformas-items'), $nameImage);
                }

                ProformaItem::create([
                    'proforma_id' => $proforma->id,
                    'product_id' => $item['product_id'] ?? null,
                    'product_variant_id'=> isset($item['product_variant_id']) && $item['product_variant_id'] != "null" ? $item['product_variant_id'] : 0,
                    'main_image' => isset($item['main_image']) ? $item['main_image'] : '',
                    'product_name' => $item['product_name'] ?? '',
                    'size' => $item['size'] ?? '',
                    'color' => $item['color'] ?? '',
                    'custom_name' => $item['custom_name'] ?? null,
                    'custom_description' => $item['custom_description'] ?? null,
                    'custom_measurements' => $item['custom_measurements'] ?? null,
                    'custom_image' => $nameImage,
                    'quantity' => $item['quantity'],
                    'price' => $item['price'],
                ]);
            }

            return $proforma;
        });
    }

    public function update(int $id, array $data, array $items): ?Proforma
    {
        $data['items'] = $items;

        return DB::transaction(function () use ($id, $data) {
            $proforma = Proforma::with('items')->find($id);
            if (!$proforma) return null;

            $proforma->update([
                'user_id' => $data['user_id'],
                'customer_id' => $data['customer_id'],
                'status' => $data['status'] ?? $proforma->status,
                'observations' => $data['observations'] ?? $proforma->observations,
                'total' => $data['total'],
                'delivery_date' => $data['delivery_date'],
                'has_furniture_service' => $data['has_furniture_service'],
            ]);

            foreach ($data['items'] as $item) {
                $proformaItem = ProformaItem::find((isset($item['id']) ? $item['id'] : ''));

                $nameImage = '';

                if ($proformaItem) {
                    if (isset($item['custom_image'])) {
                        if ($item['custom_image'] instanceof \Illuminate\Http\UploadedFile) {
                            $image     = $item['custom_image'];
                            $nameImage = date('YmdHis') .'.'. $image->getClientOriginalExtension();
                            $image->move(public_path('img/proformas-items'), $nameImage);

                            \File::delete(public_path('img/proformas-items/'. $proformaItem->custom_image));
                        }
                        else {
                            $nameImage = $proformaItem->custom_image;
                        }
                    }

                    $proformaItem->update([
                        'proforma_id' => $proforma->id,
                        'product_id' => $item['product_id'] ?? null,
                        'product_variant_id'=> isset($item['product_variant_id']) && $item['product_variant_id'] != "null" ? $item['product_variant_id'] : 0,
                        'main_image' => isset($item['main_image']) ? $item['main_image'] : '',
                        'product_name' => $item['product_name'] ?? '',
                        'size' => $item['size'] ?? '',
                        'color' => $item['color'] ?? '',
                        'custom_name' => $item['custom_name'] ?? null,
                        'custom_description' => $item['custom_description'] ?? null,
                        'custom_measurements' => $item['custom_measurements'] ?? null,
                        'custom_image' => $nameImage,
                        'quantity' => $item['quantity'],
                        'price' => $item['price'],
                    ]);
                }
                else {
                    if (isset($item['custom_image'])) {
                        $image     = $item['custom_image'];
                        $nameImage = date('YmdHis') .'.'. $image->getClientOriginalExtension();
                        $image->move(public_path('img/proformas-items'), $nameImage);
                    }

                    ProformaItem::create([
                        'proforma_id' => $proforma->id,
                        'product_id' => $item['product_id'] ?? null,
                        'product_variant_id'=> isset($item['product_variant_id']) && $item['product_variant_id'] != "null" ? $item['product_variant_id'] : 0,
                        'main_image' => isset($item['main_image']) ? $item['main_image'] : '',
                        'product_name' => $item['product_name'] ?? '',
                        'size' => $item['size'] ?? '',
                        'color' => $item['color'] ?? '',
                        'custom_name' => $item['custom_name'] ?? null,
                        'custom_description' => $item['custom_description'] ?? null,
                        'custom_measurements' => $item['custom_measurements'] ?? null,
                        'custom_image' => $nameImage,
                        'quantity' => $item['quantity'],
                        'price' => $item['price'],
                    ]);
                }
            }

            return $proforma;
        });
    }

    public function delete(int $id): bool
    {
        $proforma = Proforma::find($id);
        $proforma->status = "cancelado";

        return $proforma ? $proforma->save() : false;
    }

    public function confirmOrder(int $id, $data)
    {
        $proforma = Proforma::find($id);
        $proforma->status = "Confirmado";
        $proforma->advance_amount = $data->advance_amount;
        $proforma->save();

        return $proforma;
    }

    public function convertToSale($items, $id, $userId, $branchId, $cashRegisterId)
    {
        return DB::transaction(function () use ($items, $id, $userId, $branchId, $cashRegisterId) {
            $proforma = Proforma::find($id);

            if ($proforma->status == 'Atendido' || $proforma->status == 'Cancelado') {
                return response()->json(['message' => 'La proforma ya fue Atendida o Cancelada.'], 400);
            }

            // Crear la venta
            $sale = Sale::create([
                'cash_register_id' => $cashRegisterId,
                'customer_id' => $proforma->customer_id,
                'branch_id' => $branchId,
                'user_id'     => $userId,
                'status'      => $proforma->total - $proforma->advance_amount == 0 ? 'paid' : 'unpaid',
                'total' => $proforma->total,
                'advance_payment' => $proforma->advance_amount,
                'new_date' => date('Y-m-d H:i:s'),
                'completed_date' => date('Y-m-d H:i:s'),
            ]);

            foreach ($items as $item) {
                $branchProductStock = $this->branchProductStockService->getById($item['selected_branch_stock_id']);

                $proformaItem = ProformaItem::find($item['id']);
                $proformaItem->sold_from_branch_id = $branchProductStock->branch_id;
                $proformaItem->save();

                $this->branchProductStockService->updateStockById($item['selected_branch_stock_id'], -$item['quantity']);
            }

            // Actualizar proforma
            $proforma->status = 'Atendido';
            $proforma->sale_id = $sale->id;
            $proforma->user_id_served = $userId;
            $proforma->save();

            return $proforma;
        });
    }

    public function deleteItem(int $id): bool
    {
        $proformaItem = ProformaItem::find($id);

        if ($proformaItem->custom_image)
            \File::delete(public_path('img/proformas-items/'. $proformaItem->custom_image));

        return $proformaItem->delete() ? true : false;
    }
}
