<?php

namespace App\Services;

use App\Models\Purchase;
use App\Models\PurchaseDetail;
use App\Services\{ BranchProductStockService, ProductService, ProductVariantService };
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class PurchaseService
{
    protected $branchProductStockService;
    protected $productService;
    protected $productVariantService;

    public function __construct(BranchProductStockService $branchProductStockService, ProductService $productService, ProductVariantService $productVariantService)
    {
        $this->branchProductStockService = $branchProductStockService;
        $this->productService = $productService;
        $this->productVariantService = $productVariantService;
    }

    public function list($request)
    {
        $purchases = Purchase::with('supplier');

        if ($request->search) {
            $purchases = $purchases->where('invoice_number', 'like', '%'. $request->search .'%');
        }
        else {
            if ($request->start_date && $request->end_date) {
                $startDate = $request->start_date .' 00:00:00';
                $endDate = $request->end_date .' 23:59:59';
    
                $purchases = $purchases->whereBetween('created_at', [$startDate, $endDate]);
            }
            
            if ($request->order_status) {
                $purchases = $purchases->where('order_status', $request->order_status);
            }

            if ($request->payment_status) {
                $purchases = $purchases->where('payment_status', $request->payment_status);
            }

            if ($request->supplier_id) {
                $purchases = $purchases->where('supplier_id', $request->supplier_id);
            }
        }

        if ($request->action && $request->action == 'export') {
            return $purchases->orderBy('created_at', 'desc')->get();
        }

        return $purchases->orderBy('created_at', 'desc')->paginate(20);
    }
    
    public function getById(int $id): ?Purchase
    {
        return Purchase::with(['supplier', 'details.product', 'payments'])->find($id);
    }

    public function create(array $data) 
    {
        return DB::transaction(function () use ($data) {
            $data['new_at'] = Carbon::now();
            $data['unpaid_at'] = Carbon::now();
            $data['total_amount'] = $this->calculateTotalAmount($data['purchaseDetails']);
            $purchase = Purchase::create($data);

            foreach ($data['purchaseDetails'] as $detail) {
                PurchaseDetail::create([
                    'purchase_id' => $purchase->id,
                    'product_id' => $detail['productId'],
                    'variant_id' => $detail['variantId'],
                    'product_name' => $detail['name'],
                    'size' => $detail['size'],
                    'color' => $detail['color'],
                    'quantity' => $detail['quantity'],
                    'purchase_price' => $detail['purchase_price'],
                    'subtotal' => $detail['quantity'] * $detail['purchase_price'],
                ]);

                /*if ($detail['variantId'] == '')
                    $this->productService->updateStock($detail['productId'], $detail['quantity'], $detail['price']);
                else 
                    $this->productVariantService->updateStock($detail['variantId'], $detail['quantity'], $detail['price']);*/
            }

            return $purchase->load('details.product');
        });
    }

    public function calculateTotalAmount($purchaseDetails)
    {
        $total_amount = 0;

        foreach ($purchaseDetails as $detail) {
            $total_amount = $total_amount + ($detail['quantity'] * $detail['purchase_price']);
        }

        return $total_amount;
    }

    public function update(array $data, $purchaseId)
    {
        return DB::transaction(function () use ($data, $purchaseId) {
            $data['total_amount'] = $this->calculateTotalAmount($data['purchaseDetails']);

            $purchase = Purchase::find($purchaseId);

            foreach ($data['purchaseDetails'] as $detail) {
                $purchaseDetail = PurchaseDetail::find($detail["id"]);

                if($purchaseDetail){
                    $purchaseDetail->update([
                        'quantity' => $detail['quantity'],
                        'purchase_price' => $detail['purchase_price'],
                        'subtotal' => $detail['quantity'] * $detail['purchase_price'],
                    ]);
                }
                else{
                    PurchaseDetail::create([
                        'purchase_id' => $purchase->id,
                        'product_id' => $detail['productId'],
                        'variant_id' => $detail['variantId'],
                        'product_name' => $detail['name'],
                        'size' => $detail['size'],
                        'color' => $detail['color'],
                        'quantity' => $detail['quantity'],
                        'purchase_price' => $detail['purchase_price'],
                        'sale_price' => $data['order_status'] == 'stored' ? $detail['price'] : 0,
                        'subtotal' => $detail['quantity'] * $detail['purchase_price'],
                    ]);
                }
  
                if (isset($data['order_status']) && $data['order_status'] == 'stored') {
                    if ($detail['variantId'] == ''){
                        $this->productService->updateStock($detail, "purchase");
                        $this->branchProductStockService->updateStockPurchase(1, $detail['productId'], null, $detail['quantity']);
                    }
                    else{
                        $this->productVariantService->updateStock($detail, "purchase");
                        $this->branchProductStockService->updateStockPurchase(1, $detail['productId'], $detail['variantId'], $detail['quantity']);
                    }
                }
            } 

            if (isset($data['order_status']) && $data['order_status'] == 'confirmed') {
                $purchase["confirmed_at"] = date('Y-m-d H:i:s');
            }
            else if (isset($data['order_status']) && $data['order_status'] == 'to_pickup') {
                $purchase["to_pickup_at"] = date('Y-m-d H:i:s');
            }
            else if (isset($data['order_status']) && $data['order_status'] == 'stored') {
                $purchase["stored_at"] = date('Y-m-d H:i:s');
            }

            $purchase->update($data);

            return $purchase->load('details.product');
        });
    } 

    public function updateStatus($id, $status): ?Purchase
    {
        $purchase = Purchase::find($id);
        if ($status == 'confirmed') {
            $purchase["confirmed_at"] = date('Y-m-d H:i:s');
        }
        else if ($status == 'to_pickup') {
            $purchase["to_pickup_at"] = date('Y-m-d H:i:s');
        }
        else if ($status == 'stored') {
            $purchase["stored_at"] = date('Y-m-d H:i:s');
        }

        $purchase["order_status"] = $status;
        $purchase->update();

        return $purchase;
    }

    public function delete(int $id): bool
    {
        return DB::transaction(function () use ($id) {
            $purchase = Purchase::find($id);

            $purchaseDetails =  PurchaseDetail::where('purchase_id', $id)->get();

            foreach ($purchaseDetails as $detail) {
                $purchaseDetail = PurchaseDetail::find($detail->id);
                $purchaseDetail->delete();
            }

            return $purchase->delete() ? true : false;
        });
    }

    public function updateAdvancePayment(int $id, $amount)
    {
        $purchase = $this->getByid($id);
        $purchase->advance_payment = $purchase->advance_payment + $amount;

        if ($purchase->advance_payment == $purchase->total_amount) {
            $purchase->paid_at = date('Y-m-d H:i:s');
            $purchase->payment_status = 'paid';
        }
        else {
            $purchase->partial_at = date('Y-m-d H:i:s');
            $purchase->payment_status = 'partial';
        }

        $purchase->save();
    }

    public function toPayList()
    {
        return Purchase::with('supplier')->where('status', 'unpaid')->get();
    }

    public function deleteDetail(int $purchaseDetailId): bool
    {
        return DB::transaction(function () use ($purchaseDetailId) {
            $purchaseDetail = PurchaseDetail::find($purchaseDetailId);

            $purchase = Purchase::find($purchaseDetail->purchase_id);
            $purchase->total_amount = $purchase->total_amount - $purchaseDetail->subtotal;
            $purchase->save();

            return $purchaseDetail->delete() ? true : false;
        });
    }
}
