<?php

namespace App\Services;

use App\Models\Sale;
use App\Models\SaleDetail;
use App\Services\{ ProductService, ProductVariantService };
use Illuminate\Support\Facades\DB;

class SaleService
{
    protected $branchProductStockService;
    protected $productService;
    protected $productVariantService;

    public function __construct(BranchProductStockService $branchProductStockService, ProductService $productService, ProductVariantService $productVariantService)
    {   
        $this->branchProductStockService = $branchProductStockService;
        $this->productService = $productService;
        $this->productVariantService = $productVariantService;
    }

    public function list($request)
    {
        $sales = Sale::with('customer');

        if ($request->start_date && $request->end_date) {
            $startDate = $request->start_date .' 00:00:00';
            $endDate = $request->end_date .' 23:59:59';

            $sales = $sales->whereBetween('created_at', [$startDate, $endDate]);
        }

        if ($request->clientData) {
            $sales = $sales->whereHas('customer', function($query) use($request){
                $query->select("id", "first_name", "last_name", DB::Raw("CONCAT(first_name, ' ', last_name, ' ', middle_name) AS fullName"))->having("fullName", "like", "%". $request->clientData ."%");
            });
        }

        if ($request->status) {
            $sales = $sales->where('status', $request->status);
        }
 
        if ($request->action && $request->action == 'export') {
            return $sales->orderBy('created_at', 'desc')->get();
        }

        return $sales->orderBy('created_at', 'desc')->paginate(20);
    }

    public function create(array $data)
    {
        $data['total'] = $data['total_amount'] = $this->calculateTotalAmount($data['saleDetails']);
        
        return DB::transaction(function () use ($data) {
            // Create sale
            $sale = Sale::create([
                'cash_register_id' => $data['cash_register_id'],
                'branch_id' => $data['branch_id'],
                'user_id' => $data['user_id'],
                'customer_id' => $data['customer_id'],
                'total' => $data['total'],
                'advance_payment' => $data['status'] == 'unpaid' ? $data['advance_payment'] : $data['total'],
                'status' => $data['status'],
                'completed_date' => $data['status'] == 'unpaid' ? null : date('Y-m-d H:i:s'),
                'new_date' => date('Y-m-d H:i:s')
            ]);

            // Insert sale details
            foreach ($data['saleDetails'] as $detail) {
                SaleDetail::create([
                    'sale_id' => $sale->id,
                    'product_id' => $detail['productId'],
                    'variant_id' => $detail['variantId'],
                    'product_name' => $detail['name'],
                    'size' => $detail['size'], 
                    'color' => $detail['color'],
                    'quantity' => $detail['quantity'],
                    'price' => $detail['price'],
                    'subtotal' => $detail['quantity'] * $detail['price'],
                ]);

                if ($detail['variantId'] == '')
                    $this->branchProductStockService->updateStockPurchase($data['branch_id'], $detail['productId'], null, -$detail['quantity']);
                else 
                    $this->branchProductStockService->updateStockPurchase($data['branch_id'], $detail['productId'], $detail['variantId'], -$detail['quantity']);
            }

            return $sale;
        });
    }

    public function getByid($id): Sale
    {
        return Sale::with(['details.product', 'details.product_variant.color', 'details.product_variant.size', 'customer', 'branch', 'user'])->find($id); 
    }

    public function calculateTotalAmount($purchaseDetails)
    {
        $total_amount = 0;

        foreach ($purchaseDetails as $detail) {
            $total_amount = $total_amount + ($detail['quantity'] * $detail['price']);
        }

        return $total_amount;
    }

    public function update(Sale $sale, array $data)
    {
        return DB::transaction(function () use ($sale, $data) {
            // Update sale
            $sale->update([
                'user_id' => $data['user_id'],
                'customer_id' => $data['customer_id'],
                'sale_date' => $data['sale_date'],
                'total' => $data['total'],
            ]);

            // Delete old details and insert new ones
            $sale->saleDetails()->delete();

            foreach ($data['saleDetails'] as $detail) {
                SaleDetail::create([
                    'sale_id' => $sale->id,
                    'product_id' => $detail['product_id'],
                    'quantity' => $detail['quantity'],
                    'price' => $detail['price'],
                    'subtotal' => $detail['quantity'] * $detail['price'],
                ]);
            }

            return $sale;
        });
    }

    public function delete(int $id): bool
    {
        return DB::transaction(function () use ($id) {
            $sale = Sale::find($id);
            $sale->status = 'canceled';
            $sale->canceled_date = date('Y-m-d H:i:s');

            $saleDetails =  SaleDetail::where('sale_id', $id)->get();

            foreach ($saleDetails as $detail) {
                $this->productService->updateStockNotVariant($detail->product_id, $detail->quantity, $detail->unit_price);
            }

            return $sale->save() ? true : false;
        });
    }

    public function updateAdvancePayment(int $id, $amount)
    {
        $sale = $this->getByid($id);
        $sale->advance_payment = $sale->advance_payment + $amount;

        if ($sale->advance_payment == $sale->total) {
            $sale->completed_date = date('Y-m-d H:i:s');
            $sale->status = 'paid';
        }

        $sale->save();
    }

    public function toCollectList()
    {
        return Sale::with('customer')->where('status', 'unpaid')->get();
    }

    public function calculateTotalSale($cashRegisterId)
    {
        return Sale::where('cash_register_id', $cashRegisterId)->get()->sum('advance_payment');
    }
}
